<?php

namespace App\Models;

use \DateTimeInterface;
use App\Support\HasAdvancedFilter;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class VisitRoute extends Model
{
    use HasFactory;
    use HasAdvancedFilter;
    use SoftDeletes;

    public const JOURNEY_TYPE_RADIO = [
        'journey' => 'Journey',
        'return'  => 'return',
    ];

    public const VEHICLE_TYPE_RADIO = [
        'personal' => 'Personal Vehicle',
        'public'   => 'Public Transport',
    ];

    public $table = 'visit_routes';

    public $orderable = [
        'id',
        'from',
        'to',
        'journey_distance',
        'journey_fare',
        'vehicle_type',
        'journey_type',
        'note',
        'route_cost',
        'user.name',
        'reimbursement.amount',
    ];

    public $filterable = [
        'id',
        'from',
        'to',
        'journey_distance',
        'journey_fare',
        'vehicle_type',
        'journey_type',
        'note',
        'route_cost',
        'user.name',
        'reimbursement.amount',
    ];

    protected $dates = [
        'created_at',
        'updated_at',
        'deleted_at',
    ];

    protected $fillable = [
        'from',
        'to',
        'journey_distance',
        'journey_fare',
        'vehicle_type',
        'journey_type',
        'note',
        'route_cost',
        'user_id',
        'reimbursement_id',
    ];

    public function getVehicleTypeLabelAttribute($value)
    {
        return static::VEHICLE_TYPE_RADIO[$this->vehicle_type] ?? null;
    }

    public function getJourneyTypeLabelAttribute($value)
    {
        return static::JOURNEY_TYPE_RADIO[$this->journey_type] ?? null;
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function reimbursement()
    {
        return $this->belongsTo(VisitReimbursement::class);
    }

    protected function serializeDate(DateTimeInterface $date)
    {
        return $date->format('Y-m-d H:i:s');
    }

    #acesseor route_description
    public function getRouteDescriptionAttribute()
    {
        return $this->from . ' - ' . $this->to;
    }

    #acesseor route_vehicle_journey_note
    public function getRouteVehicleJourneyNoteAttribute()
    {
        return $this->route_description . ' - ' . $this->vehicle_type_label . ' - ' . $this->journey_type_label . ' - ' . $this->note;
    }

    #acesseor route_vehicle_journey_note_html
    public function getRouteVehicleJourneyNoteHtmlAttribute()
    {
        $arr = [
            // 'route' => $this->route_description,
            'vehicle' => $this->vehicle_type_label,
            // 'journey' => $this->journey_type_label,
            'note' => $this->note,
        ];
        $html = '';
        foreach ($arr as $key => $value) {
            if ($value) {
                $html .= '<span class="text-gray-500 text-[0.8em]">' . /* ucfirst($key) . ': '. */ '</span> <span class="text-black">' . $value . ' </span> <br>';
            }
        }
        return $html;
    }
}
