<?php

namespace App\Models;

use \DateTimeInterface;
use App\Models\UserAlert;
use App\Support\HasAdvancedFilter;
use Carbon\Carbon;
use Hash;
use Illuminate\Contracts\Translation\HasLocalePreference;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class User extends Authenticatable implements HasLocalePreference
{
    use HasFactory;
    use HasAdvancedFilter;
    use Notifiable;
    use SoftDeletes;
    use \Laravel\Sanctum\HasApiTokens;
    use \Staudenmeir\EloquentHasManyDeep\HasTableAlias;
    use \Staudenmeir\EloquentHasManyDeep\HasRelationships;
    // use \Staudenmeir\EloquentHasManyDeep\HasManyDeep;
    use \MBarlow\Megaphone\HasMegaphone;

    // admin,user,circle-officer,zonal-officer,engineer
    // public static $userGates = [
    //     'admin' => function (User $user) {
    //         return $user->isAdmin();
    //     },
    //     'user' => function (User $user) {
    //         return $user->isUser();
    //     },
    //     'circle-officer' => function (User $user) {
    //         return $user->isCircleOfficer();
    //     },
    //     'zonal-officer' => function (User $user) {
    //         return $user->isZonalOfficer();
    //     },
    //     'engineer' => function (User $user) {
    //         return $user->isEngineer();
    //     },
    // ];

    // public static function getUserGates()
    // {
    //     return self::$userGates;
    // }

    public $table = 'users';

    public $orderable = [
        'id',
        'name',
        'email',
        'email_verified_at',
        'locale',
    ];

    public $filterable = [
        'id',
        'name',
        'email',
        'phone',
        'username',
        'employee_code',
        'email_verified_at',
        'roles.title',
        'locale',
    ];

    protected $hidden = [
        'remember_token',
        'password',
    ];

    protected $fillable = [
        'name',
        'email',
        'password',
        // 'locale',
        'phone',
        'username',
        'employee_code',
        'is_active',
    ];

    protected $dates = [
        'email_verified_at',
        'created_at',
        'updated_at',
        'deleted_at',
    ];

    // with roles
    protected $with = ['roles'];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['role', 'role_primary', 'name_role'];

    public function getIsAdminAttribute()
    {
        return $this->roles()->where('title', 'Admin')->exists();
    }

    public function scopeAdmins()
    {
        return $this->whereHas('roles', fn ($q) => $q->where('title', 'Admin'));
    }

    public function scopeAttendables()
    {
        return $this->whereHas('roles', fn ($q) => $q->whereIn('roles.id', [
            // Role::ROLE_ADMIN_ID,
            Role::ROLE_USER_ID,
            Role::ROLE_CIRCLE_OFFICER_ID,
            Role::ROLE_ENGINEER_HEAD_ID,
            Role::ROLE_ENGINEER_ID,
            // Role::ROLE_CLIENT_OM_ID,
            Role::ROLE_CLIENT_ZO_ID,
        ]));
    }

    // engineersAndCircleOfficers
    public function scopeEngineersAndCircleOfficers()
    {
        return $this->whereHas('roles', fn ($q) => $q->whereIn('roles.id', [
            // Role::ROLE_ADMIN_ID,
            // Role::ROLE_USER_ID,
            Role::ROLE_CIRCLE_OFFICER_ID,
            Role::ROLE_ENGINEER_HEAD_ID,
            Role::ROLE_ENGINEER_ID,
            // Role::ROLE_CLIENT_OM_ID,
            // Role::ROLE_CLIENT_ZO_ID,
        ]));
    }

    public function alerts()
    {
        return $this->belongsToMany(UserAlert::class)->withPivot('seen_at');
    }

    public function preferredLocale()
    {
        return $this->locale;
    }

    public function getEmailVerifiedAtAttribute($value)
    {
        return $value ? Carbon::createFromFormat('Y-m-d H:i:s', $value)->format(config('project.datetime_format')) : null;
    }

    public function setPasswordAttribute($input)
    {
        if ($input) {
            $this->attributes['password'] = Hash::needsRehash($input) ? Hash::make($input) : $input;
        }
    }

    public function roles()
    {
        return $this->belongsToMany(Role::class);
    }

    // role_primary
    public function getRolePrimaryAttribute()
    {
        $role = $this->roles->first();
        return $role ? $role->title : null;
    }

    // role
    public function getRoleAttribute()
    {
        return $this->role_primary;
    }

    public function role_ids()
    {
        return $this->roles()->select('id')->pluck('id')->toArray();
    }

    protected function serializeDate(DateTimeInterface $date)
    {
        return $date->format('Y-m-d H:i:s');
    }

    // public static function getApiUser(): ?User
    // {
    //     return Auth::user();
    // }
    public static function getAuthUser(): ?User
    {
        return Auth::user();
    }


    public static function userInfoResponse(User $user)
    {
        $uInfo = Arr::only($user->toArray(), [
            'id', 'name',
            'email',
            'phone',
            'username',
            'employee_code',
        ]);
        $uInfo['userId'] = $user->id;
        $uInfo['roles'] = $user->roles->map(fn ($r) => $r->title)->toArray();
        $uInfo['role_text'] = join(", ", $uInfo['roles']);
        return $uInfo;
    }

    public function attendances()
    {
        return $this->hasMany(EmployeeAttendanceLog::class);
    }
    public function regions()
    {
        return $this->belongsToMany(Region::class);
    }
    public function client_branches()
    {
        $region_ids = $this->regions()->select('id')->pluck('id')->toArray();
        // return ClientBranch::whereIn('id', $region_ids);
        return ClientBranch::whereIn('region_id', $region_ids);
    }


    // name_role
    public function getNameRoleAttribute()
    {
        return $this->name . " (" . $this->role . ")";
    }


    public function tasks()
    {
        $loadRelations = [
            'tag',
            'status',
            'assignedTo',
            'clientGroup',
            'client',
            'clientBranch',
            'circle.engineer',
        ];
        if ($this->roles[0]->id == 3) { // Circle Officer

           $client_branch_ids = $this->clientBranchesCO()
            ->select('client_branches.id')->get()->pluck('id')->toArray();
            return Task::with($loadRelations)->whereIn('client_branch_id', $client_branch_ids)->latest()->get();
        }
        if ($this->roles[0]->id == 4) { // Engineer Head
            $client_branch_ids = $this->client_branches()->select('id')->get()->pluck('id')->toArray();
            return Task::with($loadRelations)->whereIn('client_branch_id', $client_branch_ids)->latest()->get();
        }
        if ($this->roles[0]->id == 5) { // Engineer
            return Task::with($loadRelations)->where('assigned_to_id', $this->id)->latest()->get();
        }
    }
    public function tasksQ()
    {
        $loadRelations = [
            'tag',
            'status',
            'assignedTo',
            'clientGroup',
            'client',
            'clientBranch',
            'circle.engineer',
        ];
        if ($this->roles[0]->id == 3) { // Circle Officer

           $client_branch_ids = $this->clientBranchesCO()
            ->select('client_branches.id')->get()->pluck('id')->toArray();
            return Task::with($loadRelations)->whereIn('client_branch_id', $client_branch_ids)->latest();
        }
        if ($this->roles[0]->id == 4) { // Engineer Head
            $client_branch_ids = $this->client_branches()->select('id')->get()->pluck('id')->toArray();
            return Task::with($loadRelations)->whereIn('client_branch_id', $client_branch_ids)->latest();
        }
        if ($this->roles[0]->id == 5) { // Engineer
            return Task::with($loadRelations)->where('assigned_to_id', $this->id)->latest();
        }
    }



    public function engineers()
    {

        return $this->hasManyDeep(User::class, ['App\\Models\\CircleUser as cu1', Circle::class, 'App\\Models\\CircleUser as cu2'])
            ->where('cu2.user_id', '<>', $this->id);

        // return $this->hasManyDeep(User::class, ['App\\Models\\RegionUser as ru1 ', Region::class, 'App\\Models\\RegionUser as ru2'])
        //     ->where('ru2.user_id', '<>', $this->id);


        // $region_ids = $this->regions()->select('id')->pluck('id')->toArray();

        // return $this->hasManyDeepFromRelations(
        //     $this->regions(),
        //     (new Region())->setAlias('reg')->engineer()
        // );

        // return $this->hasManyDeep(
        //     User::class,
        //     ['region_user'], // Intermediate models and tables, beginning at the far parent (User).
        //     [
        //         'user_id', // Foreign key on the "role_user" table.
        //         'region_id',      // Foreign key on the "roles" table (local key).
        //         'id3'  // Foreign key on the "permissions" table.
        //     ],
        //     [
        //         'id',      // Local key on the "users" table.
        //         'region_id', // Local key on the "role_user" table (foreign key).
        //         'id'       // Local key on the "roles" table.
        //     ]
        // );
        // return $this->hasManyDeep(User::class, [DB::Raw('region_user as ru1'), Region::class, DB::raw('region_user as ru2')]);
        // return $this->hasManyDeep(User::class, [DB::Raw('region_user as ru1'), Region::class, DB::raw('region_user as ru2')]);
    }


    public function isAdmin()
    {
        return $this->roles->where('id', 1)->count() > 0;
    }

    public function isUser()
    {
        return $this->roles->where('id', 2)->count() > 0;
    }

    public function isCircleOfficer()
    {
        // return $this->roles->where('id', 3)->count() > 0;
        return $this->roles->whereIn('id', [3, 4])->count() > 0;
    }

    public function isHeadEngineer()
    {
        // return $this->roles->where('id', 4)->count() > 0;
        return $this->roles->whereIn('id', [3, 4])->count() > 0;
    }

    public function isEngineer()
    {
        return $this->roles->where('id', 5)->count() > 0;
    }
    public function isAnyEngineer()
    {
        // return $this->roles->whereIn('id', [4, 5])->count() > 0;
        return $this->roles->whereIn('id', [3, 4, 5])->count() > 0;

    }

    public function isClientOM()
    {
        return $this->roles->where('id', 6)->count() > 0;
    }

    public function isZonalOfficer()
    {
        return $this->roles->where('id', 7)->count() > 0;
    }


    public function clientGroups()
    {
        return $this->hasMany(ClientGroup::class, 'operation_manager_id', 'id');
    }

    public function employeeDetails()
    {
        return $this->hasOne(EmployeeInfo::class);
    }

    public function paySlips()
    {
        return $this->hasMany(MonthlyPayslip::class);
    }


    private static $userMenu = [
        [
            "icon" => "analyticsOutline",
            "name" => "Dashboard",
            "route" => "/dummy/dashboard",
            "code" => "dashboard"
        ],
        [
            "icon" => "personOutline",
            "name" => "Profile",
            "route" => "/profile",
            "code" => "profile"
        ],
        [
            "icon" => "peopleOutline",
            "name" => "Engineers",
            "route" => "/engineers",
            "code" => "engineers"
        ],
        [
            "icon" => "calendarOutline",
            "name" => "Tasks",
            "route" => "/tasks",
            "code" => "tasks"
        ],
        [
            "icon" => "todayOutline",
            "name" => "Attendance",
            "route" => "/attendance",
            "code" => "attendance"
        ],
        [
            "icon" => "calendarClearOutline",
            "name" => "Leaves",
            "route" => "/dummy/leaves",
            "code" => "leaves"
        ],
        [
            "icon" => "hourglassOutline",
            "name" => "Time Off",
            "route" => "/dummy/time-off",
            "code" => "time-off"
        ],
        [
            "icon" => "locationOutline",
            "name" => "Visits",
            "route" => "/my-visits",
            "code" => "visits"
        ],
        [
            "icon" => "cardOutline",
            "name" => "Payroll",
            "route" => "/payrolls",
            "code" => "payroll"
        ],
        [
            "icon" => "walletOutline",
            "name" => "Reimbursement",
            "route" => "/reimbursement",
            "code" => "reimbursement"
        ],
        [
            "icon" => "barChartOutline",
            "name" => "Reports",
            "route" => "/dummy/reports",
            "code" => "reports"
        ],
        [
            "icon" => "documentTextOutline",
            "name" => "Time Sheet",
            "route" => "/dummy/time-sheet",
            "code" => "time-sheet"
        ]
    ];



    public function getAppMenu()
    {
        // roles and menu map array
        $roleMenuMap = [
            'admin' => [
                // 'dashboard',
                'profile',
                // 'engineers',
                // 'tasks',
                'attendance',
                // 'leaves',
                'time-off',
                // 'visits',
                'payroll',
                // 'reimbursement',
                'reports',
                // 'time-sheet'
            ],
            'user' => [
                // 'dashboard',
                'profile',
                // 'engineers',
                // 'tasks',
                'attendance',
                'leaves',
                'time-off',
                // 'visits',
                'payroll',
                // 'reimbursement',
                'reports',
                // 'time-sheet'
            ],
            'circle_officer' => [
                // 'dashboard',
                'profile',
                'engineers',
                'tasks',
                'attendance',
                'leaves',
                'time-off',
                'visits',
                'payroll',
                'reimbursement',
                'reports',
                // 'time-sheet'
            ],
            'head_engineer' => [
                // 'dashboard',
                'profile',
                'engineers',
                'tasks',
                'attendance',
                'leaves',
                'time-off',
                'visits',
                'payroll',
                'reimbursement',
                'reports',
                // 'time-sheet'
            ],
            'engineer' => [
                // 'dashboard',
                'profile',
                'engineers',
                'tasks',
                'attendance',
                'leaves',
                'time-off',
                'visits',
                'payroll',
                'reimbursement',
                'reports',
                // 'time-sheet'
            ],
            'client_om' => [
                // 'dashboard',
                // 'profile',
                // 'engineers',
                // 'tasks',
                // 'attendance',
                // 'leaves',
                // 'time-off',
                // 'visits',
                // 'payroll',
                // 'reimbursement',
                // 'reports',
                // 'time-sheet'
            ],
            'zonal_officer' => [
                // 'dashboard',
                'profile',
                'engineers',
                'tasks',
                'attendance',
                'leaves',
                'time-off',
                'visits',
                'payroll',
                'reimbursement',
                'reports',
                // 'time-sheet'
            ],

        ];

        $menu = collect();
        $allMenu = collect(self::$userMenu);
        if ($this->isAdmin()) {

            $menu = $allMenu->whereIn('code', $roleMenuMap['admin']);
        } else if ($this->isUser()) {
            $menu = $allMenu->whereIn('code', $roleMenuMap['user']);
        } else if ($this->isCircleOfficer()) {
            $menu = $allMenu->whereIn('code', $roleMenuMap['circle_officer']);
        } else if ($this->isHeadEngineer()) {
            $menu = $allMenu->whereIn('code', $roleMenuMap['head_engineer']);
        } else if ($this->isEngineer()) {
            $menu = $allMenu->whereIn('code', $roleMenuMap['engineer']);
        } else if ($this->isClientOM()) {
            $menu = $allMenu->whereIn('code', $roleMenuMap['client_om']);
        } else if ($this->isZonalOfficer()) {
            $menu = $allMenu->whereIn('code', $roleMenuMap['zonal_officer']);
        }
        return $menu->values()->toArray();
    }

    // hasMany circles [circle_officer_id]
    public function circles()
    {
        return $this->hasMany(Circle::class,  'circle_officer_id');
    }

    // hasMany zones [zonal_officer]
    public function zones()
    {
        $rel = $this->hasMany(Zone::class,  'zonal_officer_id');
        if ($this->isZonalOfficer()) {
            return $rel;
        }
        return $this->hasMany(Zone::class,  'zonal_officer_id');
    }


    public static function getWebUser(): ?User
    {
        $user = Auth::guard('web')->user();
        return $user;
    }

    public static function getApiUser(): ?User
    {
        // $user = Auth::guard('api')->user();
        // return $user;
        return self::getAuthUser();
    }



    /**
     * Get the product requisitions requested by the user.
     */
    public function productRequisitionsRequestedByMe()
    {
        return $this->morphMany(ProductRequisition::class, 'requested_by');
    }

    /**
     * Get the product requisitions accepted by the user.
     */
    public function productRequisitionsAcceptedByMe()
    {
        return $this->morphMany(ProductRequisition::class, 'accepted_by');
    }


    public function clientBranchesCO()
    {
        return $this->hasManyThrough(ClientBranch::class, Circle::class, 'circle_officer_id', 'circle_id');
    }


}
