<?php

namespace App\Models;

use \DateTimeInterface;
use App\Support\HasAdvancedFilter;
use App\Traits\Auditable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Storage;

class TaskEventLog extends Model
{
    use HasFactory;
    use HasAdvancedFilter;
    use SoftDeletes;
    use Auditable;

    public const EVENT_TYPE_TASK_CREATED                =   'task_created';
    public const EVENT_TYPE_STATUS_CHANGED              =   'status_changed';
    public const EVENT_TYPE_ASSIGNEE_CHANGED            =   'assignee_changed';
    public const EVENT_TYPE_TAGS_CHANGED                =   'tags_changed';
    public const EVENT_TYPE_SIGNED_IN_TO_BRANCH         =   'signed_in_to_branch';
    public const EVENT_TYPE_SIGNED_OUT_FROM_BRANCH      =   'signed_out_from_branch';
    public const EVENT_TYPE_PRODUCTS_REQUESTED          =   'products_requested';
    public const EVENT_TYPE_PRODUCTS_SENT               =   'products_requested';
    public const EVENT_TYPE_TASK_CLOSED                 =   'task_closed';

    public const EVENT_TYPE_RADIO = [
        self::EVENT_TYPE_TASK_CREATED               =>  'Task Created',
        self::EVENT_TYPE_STATUS_CHANGED             =>  'Status Changed',
        self::EVENT_TYPE_ASSIGNEE_CHANGED           =>  'Assignee Changed',
        self::EVENT_TYPE_TAGS_CHANGED               =>  'Tags Changed',
        self::EVENT_TYPE_SIGNED_IN_TO_BRANCH        =>  'Signed In To Branch',
        self::EVENT_TYPE_PRODUCTS_REQUESTED         =>  'Products Requested',
        self::EVENT_TYPE_PRODUCTS_SENT              =>  'Products Sent',
        self::EVENT_TYPE_TASK_CLOSED                =>  'Task Closed',
        // self::EVENT_TYPE_SIGNED_OUT_FROM_BRANCH     =>  'Signed Out From Branch',
    ];

    public $table = 'task_event_logs';

    public $orderable = [
        'id',
        'task.name',
        'logged_by.name',
        'event_type',
        'assigned_to.name',
        'description',
    ];

    public $filterable = [
        'id',
        'task.name',
        'logged_by.name',
        'event_type',
        'assigned_to.name',
        'description',
    ];

    protected $dates = [
        'created_at',
        'updated_at',
        'deleted_at',
    ];
    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'additional' => 'array',
        'info' => 'array',
        'extras' => 'array',
    ];

    protected $fillable = [
        'task_id',
        'logged_by_id',
        'event_type',
        'assigned_to_id',
        'description',
        'additional',
        'info',
        'extras',
        'call_report_image',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['event_type_label', 'event_type_description', 'call_report_image_url'];

    public function task()
    {
        return $this->belongsTo(Task::class);
    }

    public function loggedBy()
    {
        return $this->belongsTo(User::class);
    }

    // event_type_label
    public function getEventTypeLabelAttribute($value)
    {
        return static::EVENT_TYPE_RADIO[$this->event_type] ?? null;
    }
    // event_type_description
    public function getEventTypeDescriptionAttribute($value)
    {
        if ($this->event_type == self::EVENT_TYPE_STATUS_CHANGED) {
            $text = $this->event_type_label;
            // if (Arr::has($this->info, 'old_status')) {
            //     $text .= ' from "' . Arr::get($this->info, 'old_status.name') . '"';
            // }
            if (Arr::has($this->info, 'new_status')) {
                $text .= ' from "' . Arr::get($this->info, 'new_status.name') . '"';
            }
            // $text .= ' to ' . $this->info['new_status'];
            return $text;
        }
        return $this->event_type_label;
    }

    public function assignedTo()
    {
        return $this->belongsTo(User::class);
    }

    protected function serializeDate(DateTimeInterface $date)
    {
        return $date->format('Y-m-d H:i:s');
    }


    public static function createStatusChangedEvent($task, $changedBy, $description, $oldStatus = null, $newStatus = null)
    {
        $evt = new TaskEventLog;
        $evt->task_id = $task->id;
        $evt->logged_by_id = $changedBy->id;
        $evt->event_type = self::EVENT_TYPE_STATUS_CHANGED;
        $evt->assigned_to_id = null;
        $evt->description = $description;
        $evt->additional = [];
        $evt->info = [
            'old_status' => $oldStatus,
            'new_status' => $newStatus,
        ];
        $evt->extras = [];

        $evt->save();
        if ($task->circleOfficer?->id != $changedBy->id) {
            $notification = new \MBarlow\Megaphone\Types\Important(
                'Status Changed!', // Notification Title
                $description, // Notification Body
                // 'https://example.com/link', // Optional: URL. Megaphone will add a link to this URL within the Notification display.
                // 'Read More...' // Optional: Link Text. The text that will be shown on the link button.
            );
            $task->circleOfficer?->notify($notification);
        }
    }

    public static function createTaskCreatedEvent($task, $createdBy, $description, $oldStatus = null, $newStatus = null)
    {
        $evt = new TaskEventLog;
        $evt->task_id = $task->id;
        $evt->logged_by_id = $createdBy->id;
        $evt->event_type = self::EVENT_TYPE_TASK_CREATED;
        $evt->assigned_to_id = null;
        $evt->description = $description;
        $evt->additional = [];
        $evt->info = [
            // 'old_status' => $oldStatus,
            // 'new_status' => $newStatus,
        ];
        $evt->extras = [];

        $evt->save();

        $notification = new \MBarlow\Megaphone\Types\Important(
            'Ticket Created!',
            $description,
            route('co.tasks.show', [
                'task' => $task->id,
            ]),
            'Details'
        );
        $task->circleOfficer?->notify($notification);
    }

    public static function closeTaskWithCallReport($task, $closedBy, $callReportImage, $description, $locationInfo, $oldStatus = null)
    {
        $evt = new TaskEventLog;
        $evt->task_id = $task->id;
        $evt->logged_by_id = $closedBy->id;
        $evt->event_type = self::EVENT_TYPE_TASK_CLOSED;
        $evt->assigned_to_id = null;
        $evt->description = $description;
        $evt->call_report_image = $callReportImage;
        $evt->additional = [];
        $newStatus = TaskStatus::find(TaskStatus::STATUS_CLOSED);
        $evt->info = [
            'old_status' => $oldStatus,
            'new_status' => $newStatus,
            'location_info' => $locationInfo,
        ];
        $evt->extras = [];

        $evt->save();
        $task->update([
            'status_id' => $newStatus->id,
        ]);
        if ($task->circleOfficer?->id != $closedBy->id) {
            $notification = new \MBarlow\Megaphone\Types\Important(
                'Ticket Closed!', // Notification Title
                $description, // Notification Body
                // 'https://example.com/link', // Optional: URL. Megaphone will add a link to this URL within the Notification display.
                // 'Read More...' // Optional: Link Text. The text that will be shown on the link button.
            );
            $task->circleOfficer?->notify($notification);
        }

        if ($task->circle?->zone?->zonal_officer?->id != $closedBy->id) {
            $notification = new \MBarlow\Megaphone\Types\Important(
                'Ticket Closed!', // Notification Title
                $description, // Notification Body
                // 'https://example.com/link', // Optional: URL. Megaphone will add a link to this URL within the Notification display.
                // 'Read More...' // Optional: Link Text. The text that will be shown on the link button.
            );
            $task->circle?->zone?->zonal_officer?->notify($notification);
        }
        // $task->circleOfficer?->notify($notification);
    }



    public function products()
    {
        return $this->belongsToMany(Product::class, 'product_task_event_logs');
    }

    public function all_products()
    {
        return $this->hasMany(ProductTaskEventLog::class);
    }


    public static function createProductRequestEvent($task, $changedBy, $productsSyncable)
    {
        $oldStatus = $task->status;
        $task->update([
            'status_id' => TaskStatus::STATUS_PRODUCTS_ORDERED,
        ]);
        $task->refresh();
        $newStatus = $task->status;;
        $evt = new TaskEventLog;
        $evt->task_id = $task->id;
        $evt->logged_by_id = $changedBy->id;
        $evt->event_type = self::EVENT_TYPE_PRODUCTS_REQUESTED;
        $evt->assigned_to_id = null;
        $evt->description = $description = "Products Requested";
        $evt->additional = [];
        $evt->info = [
            'old_status' => $oldStatus,
            'new_status' => $newStatus,
        ];
        $evt->extras = [];

        $evt->save();
        // $evt->products()->sync($productsSyncable);

        foreach ($productsSyncable as $pid => $prodData) {
            $productInfo = [
                'task_event_log_id' =>  $evt->id,
                'product_id' => $prodData['unlisted'] ? null : $pid,
                'quantity' => $prodData['quantity'],
                'description' => $prodData['description'],
                'unlisted' => $prodData['unlisted'],
                'info' => $prodData['info'],
            ];

            ProductTaskEventLog::create($productInfo);
        }

        if ($task->circleOfficer?->id != $changedBy->id) {
            $notification = new \MBarlow\Megaphone\Types\Important(
                'Status Changed!', // Notification Title
                $description, // Notification Body
                // 'https://example.com/link', // Optional: URL. Megaphone will add a link to this URL within the Notification display.
                // 'Read More...' // Optional: Link Text. The text that will be shown on the link button.
            );
            $task->circleOfficer?->notify($notification);
        }
    }


    // hasMany ProoductRequisition
    public function productRequisitions()
    {
        return $this->hasMany(ProductRequisition::class);
    }

    #accessor call_report_image_url
    public function getCallReportImageUrlAttribute()
    {
        return $this->call_report_image ?  url('storage/' . $this->call_report_image) : null;
    }
}
