<?php

namespace App\Models;

use \DateTimeInterface;
use App\Support\HasAdvancedFilter;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;

class Task extends Model implements HasMedia
{
    use HasFactory;
    use HasAdvancedFilter;
    use SoftDeletes;
    use InteractsWithMedia;
    use \App\Traits\Auditable;
    use \Znck\Eloquent\Traits\BelongsToThrough;

    protected static function boot()
    {
        parent::boot();
        self::saved(function ($model) {
            // (new SeoController())->updateModel($model);
        });
        self::created(function ($model) {
            // (new SeoController())->updateModel($model);
            $createdBy = auth()->user();

            TaskEventLog::createTaskCreatedEvent(
                $model,
                $createdBy,
                "New Ticket Raised By " . $createdBy->name_role,

            );
        });
        self::retrieved(function ($model) {
            // if (app()->getLocale() != config('app.fallback_locale')) {
            //     $model->localize();
            // }
        });
    }

    public const CALL_TYPE_RADIO = [
        'CM'       => 'CM',
        'PCM'      => 'PCM',
        'INS'      => 'INS',
        'REV'      => 'REV',
        'SOFTWARE' => 'SOFTWARE',
        'OTH'      => 'OTH',
    ];

    public $table = 'tasks';

    public $orderable = [
        'id',
        'name',
        'description',
        'status.name',
        'created_at',
        'due_date',
        'assigned_to.name',
        'client_group.name',
        'client_group.business_name',
        'client.name',
        'client_branch.name',
        'call_type',
        'incode',
    ];

    public $filterable = [
        'id',
        'name',
        'description',
        'status.name',
        'tag.name',
        'due_date',
        'assigned_to.name',
        'client_group.name',
        'client_group.business_name',
        'client.name',
        'client_branch.name',
        'call_type',
        'incode',
    ];

    protected $appends = [
        // 'attachment',
        'status_text',
    ];

    protected $dates = [
        'due_date',
        'created_at',
        'updated_at',
        'deleted_at',
    ];

    protected $fillable = [
        'name',
        'description',
        'status_id',
        'due_date',
        'assigned_to_id',
        'client_group_id',
        'client_id',
        'client_branch_id',
        'call_type',
        'incode',
        'created_by_id',
        //
        'is_invoiceable', 'o_invoice_id', 'amount', 'amount_w_gst', 'amount_wo_gst', 'igst_rate',
        'cgst_rate', 'sgst_rate', 'igst_amount', 'cgst_amount', 'sgst_amount', 'gst_amount', 'gst_details',
        //
        'hsn_code',
    ];

    public function status()
    {
        return $this->belongsTo(TaskStatus::class);
    }

    public function tag()
    {
        return $this->belongsToMany(TaskTag::class);
    }

    public function getAttachmentAttribute()
    {
        return $this->getMedia('task_attachment')->map(function ($item) {
            $media = $item->toArray();
            $media['url'] = $item->getUrl();

            return $media;
        });
    }

    public function getDueDateAttribute($value)
    {
        return $value ? Carbon::parse($value)->format(config('project.date_format')) : null;
    }

    public function setDueDateAttribute($value)
    {
        $this->attributes['due_date'] = $value ? Carbon::createFromFormat(config('project.date_format'), $value)->format('Y-m-d') : null;
    }

    public function assignedTo()
    {
        return $this->belongsTo(User::class);
    }
    public function createdBy()
    {
        return $this->belongsTo(User::class);
    }

    public function clientGroup()
    {
        return $this->belongsTo(ClientGroup::class);
    }

    public function client()
    {
        return $this->belongsTo(Client::class);
    }

    public function clientBranch()
    {
        return $this->belongsTo(ClientBranch::class);
    }

    // circle belongsToThrough clientBranch
    public function circle()
    {
        return $this->belongsToThrough(
            Circle::class,
            // [ClientBranch::class, Region::class],
            [ClientBranch::class],
            null,
            '',
            [Circle::class => 'circle_id']
        );
    }

    public function getCallTypeLabelAttribute($value)
    {
        return static::CALL_TYPE_RADIO[$this->call_type] ?? null;
    }

    // status_text
    public function getStatusTextAttribute()
    {
        return $this->status ? $this->status->name : null;
    }

    protected function serializeDate(DateTimeInterface $date)
    {
        return $date->format('Y-m-d H:i:s');
    }


    public function taskEventLogs()
    {
        return $this->hasMany(TaskEventLog::class);
    }

    public function taskClosedLogs()
    {
        return $this->taskEventLogs()->where('event_type', TaskEventLog::EVENT_TYPE_TASK_CLOSED);
    }

    public function isTaskClosed()
    {
        return $this->status_id == TaskStatus::STATUS_CLOSED;
    }

    public function taskClosedLogLatest()
    {
        // return $this->taskClosedLogs()->first();
        return $this->hasOne(TaskEventLog::class)
            ->where('event_type', TaskEventLog::EVENT_TYPE_TASK_CLOSED)->latestOfMany();
    }

    public function taskProductRequestLogs()
    {
        return $this->taskEventLogs()->where('event_type', TaskEventLog::EVENT_TYPE_PRODUCTS_REQUESTED);
    }

    // public function region()
    // {
    // }

    # accessor incode
    public function getIncodeAttribute()
    {
        if (!$this->id) return null;
        if ($this->clientBranch) {
            return $this->clientBranch->solid_code;
        }
        return $this->getAttribute('incode');
    }

    public function circleOfficer()
    {
        return $this->belongsToThrough(
            User::class,
            // // [ClientBranch::class, Region::class, Circle::class],
            // [Circle::class, Region::class, ClientBranch::class],
            [Circle::class, ClientBranch::class],
            null,
            '',
            [User::class => 'circle_officer_id']
        );
    }


    public function OInvoice()
    {
        return $this->belongsTo(OInvoice::class);
    }
}
