<?php

namespace App\Models;

use \DateTimeInterface;
use App\Support\HasAdvancedFilter;
use App\Traits\Auditable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class LedgerTransaction extends Model
{
    use HasFactory;
    use HasAdvancedFilter;
    use SoftDeletes;
    use Auditable;

    public const TYPE_RADIO = [
        'dr' => 'Debit',
        'cr' => 'Credit',
    ];

    public $table = 'ledger_transactions';

    public $orderable = [
        'id',
        'fin_yr.title',
        'fin_yr.alias',
        'ledger.name',
        'date',
        'amount',
        'type',
        'cr_bal_before',
        'dr_bal_before',
        'cr_bal_after',
        'dr_bal_after',
        'description',
    ];

    public $filterable = [
        'id',
        'fin_yr.title',
        'fin_yr.alias',
        'ledger.name',
        'date',
        'amount',
        'type',
        'cr_bal_before',
        'dr_bal_before',
        'cr_bal_after',
        'dr_bal_after',
        'description',
    ];

    protected $dates = [
        'date',
        'created_at',
        'updated_at',
        'deleted_at',
    ];

    protected $fillable = [
        'fin_yr_id',
        'ledger_id',
        'date',
        'amount',
        'type',
        'cr_bal_before',
        'dr_bal_before',
        'cr_bal_after',
        'dr_bal_after',
        'description',
    ];

    public function finYr()
    {
        return $this->belongsTo(FinYr::class);
    }

    public function ledger()
    {
        return $this->belongsTo(Ledger::class);
    }

    public function getDateAttribute($value)
    {
        return $value ? Carbon::createFromFormat('Y-m-d H:i:s', $value)->format(config('project.datetime_format')) : null;
    }

    public function setDateAttribute($value)
    {
        $this->attributes['date'] = $value ? Carbon::createFromFormat(config('project.datetime_format'), $value)->format('Y-m-d H:i:s') : null;
    }

    public function getTypeLabelAttribute($value)
    {
        return static::TYPE_RADIO[$this->type] ?? null;
    }

    protected function serializeDate(DateTimeInterface $date)
    {
        return $date->format('Y-m-d H:i:s');
    }
}
