<?php

namespace App\Http\Controllers\Api\V1\User;

use App\Http\Controllers\Controller;
use App\Models\EmployeeAttendanceLog;
use App\Models\ProductCategory;
use App\Models\Role;
use App\Models\Task;
use App\Models\TaskEventLog;
use App\Models\TaskStatus;
use App\Models\User;
use App\Models\VisitReimbursement;
use App\Models\VisitRoute;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class ApiAuthController extends Controller
{
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required',
            'password' => 'required',
        ]);

        $success = [];
        $user = User::where(function ($q1) use ($request) {
            $q1->where('email', $request->email)
                ->orWhere('phone', $request->email)
                ->orWhere('username', $request->email)
                ->orWhere('employee_code', $request->email)
                //
            ;
        })
            // ->where('is_active', '1')
            ->first();
        if (!$user) {
            throw ValidationException::withMessages([
                'email' => 'Credential mismatch'
            ]);
        }

        if (!Hash::check($request->password, $user->password)) {
            throw ValidationException::withMessages([
                'email' => 'Credential mismatch'
            ]);
        }


        // if not user->is_active Error
        if (!$user->is_active) {
            throw ValidationException::withMessages([
                'email' => 'Not Allowed To Login'
            ]);
        }

        // if user role_ids in array Roles::appLoginArray()

        if (!(count(array_intersect($user->role_ids(), Role::appLoginArray())) > 0)) {
            throw ValidationException::withMessages([
                'email' => 'Not Allowed To Login'
            ]);
        }


        // $authUser = User::getApiUser();
        $authUser = $user;
        $success['token'] =  $authUser->createToken('tri-staff')->plainTextToken;
        $success['name'] =  $authUser->name;
        // $success['user'] =  $authUser;
        $success['user'] = User::userInfoResponse($authUser);
        $success['user']['token'] = $success['token'];

        return $success;
        /*
        if (Auth::attempt(['email' => $request->email, 'password' => $request->password])) {
            $authUser = User::getApiUser();
            $success['token'] =  $authUser->createToken('tri-staff')->plainTextToken;
            $success['name'] =  $authUser->name;
            // $success['user'] =  $authUser;
            $success['user'] = User::userInfoResponse($authUser);
            $success['user']['token'] = $success['token'];

            return $success;
        } else {

            throw ValidationException::withMessages([
                'email' => 'Credential mismatch'
            ]);
        }
        */
    }


    public function getUser(Request $request)
    {
        return User::getApiUser();
    }


    public function getAppMenu()
    {
        $user = User::getApiUser();
        $menu = [];
        $menu['menu'] = $user->getAppMenu();
        $menu['user'] = User::userInfoResponse($user);
        return $menu;
    }


    public function logout(Request $request)
    {
        try {
            $token = $request->get('token');
            $user = User::getApiUser();
            $user->tokens()->where('id', $user->currentAccessToken()->id)->delete();
            //code...
        } catch (\Throwable $th) {
               report($th);
            //throw $th;
        }
        return [
            'success' => true,
        ];
    }


    public function changePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'new_password' => 'required|string|min:6|max:12',
            'confirm_new_password' => 'required|same:new_password',
        ]);
        $user = Auth::user();
        if (!Hash::check($request->current_password, $user->password)) {
            throw ValidationException::withMessages([
                'message' => "Current Password Is Not Correct",
                'errors' => [
                    'current_password' => "Current Password Is Not Correct"
                ],
            ]);
        }

        $user->password = Hash::make($request->new_password);
        $user->update();
        return [
            'message' => "Password changed successfully!!"
        ];
    }


    public function giveAttendance(Request $request)
    {
        $vals =   $request->validate([
            'coord_lat' => 'string|required',
            'coord_long' => 'string|required',
            'coord_accu' => 'string|nullable',
            'location' => 'string|nullable',
            'coord_data' => 'nullable',
            'geocode_data' => 'nullable',
        ]);
        $user = User::getApiUser();
        $alreadyDone = $user->attendances()->whereBetween('date', [now()->startOfDay(), now()->endOfDay()])->count() > 0;
        if (!$alreadyDone) {
            $vals['type'] = "present";
            $vals['date'] = Carbon::now()->format(config('project.date_format'));
            // $vals['date'] = '2022-10-13';
            $user->attendances()->create($vals);


            return [
                'message' => 'Attendance Taken Successfully',
            ];
        }

        throw ValidationException::withMessages([
            'message' => 'Your attendance already taken for today '
        ]);
    }



    public function tasks(Request $request)
    {

        $status = $request->input('status', "opened");
        $statusIds = [
            'opened' => [1, 2, 4, 5, 6],
            'closed' => [3],
        ];

        $user = User::getApiUser();
        if ($user->isAnyEngineer()) {
            return $user->tasks()->whereIn('status_id', Arr::get($statusIds, $status, []));
        } else {
            return [];
        }
    }

    public function taskDetailed(Request $request, Task $task)
    {
        $user = User::getApiUser();
        $notNeededStatusIds = [
            TaskStatus::STATUS_CLOSED,
            TaskStatus::STATUS_PRODUCTS_ORDERED,
            TaskStatus::STATUS_PRODUCTS_SENT,
        ];
        $task = $user->tasks()->find($task->id);
        $can_assign_to_others = $user->roles()->whereIn('id', [3, 4])->count() > 0 ? true : false;
        $can_change_status = true;
        return [
            'task' => $task,
            'can_assign_to_others' => $can_assign_to_others,
            'can_change_status' => $can_change_status,
            'statuses' => TaskStatus::select('name', 'id')->whereNotIn('id', $notNeededStatusIds)->get(),
            'product_categories' => ProductCategory::all(['description as label', 'id as value']),
            'engineers' => $can_assign_to_others
                ? ($task->circle->engineer ?? []) //$user->engineers
                : [],
        ];
    }


    public function engineers(Request $request)
    {
        $user = User::getApiUser();
        if ($user->isHeadEngineer()) {
            return $user->engineers;
        } else {
            return [];
        }
    }


    public function ChangeEngineer(Request $request, Task $task)
    {
        $request->validate([
            'engineer_id' => 'required'
        ]);
        $user = User::getApiUser();
        $task = Task::find($task->id);
        // $engineer = $user->engineers()->findOrFail($request->engineer_id);
        $engineer = $task->circle->engineer()->findOrFail($request->engineer_id);
        $updateArr = [
            'assigned_to_id' => $engineer->id,
        ];
        if ($request->status_id) {
            $updateArr['status_id'] = $request->status_id;
        }
        $task->update($updateArr);

        return $this->taskDetailed($request, $task);
    }

    public function ChangeStatus(Request $request, Task $task)
    {
        $request->validate([
            'status_id' => 'required',
            'description' => 'required',
        ]);
        $user = User::getApiUser();
        // $engineer = $user->engineers()->findOrFail($request->engineer_id);
        $oldStatus = $task->status;
        $task = Task::find($task->id);
        $updateArr = [
            'status_id' => $request->status_id,
        ];

        $task->update($updateArr);

        TaskEventLog::createStatusChangedEvent(
            $task,
            auth()->user(),
            $request->description,
            $oldStatus,
            $task->status
        );

        return $this->taskDetailed($request, $task);
    }



    public function giveTaskAttendance(Request $request, Task $task)
    {
        $vals =   $request->validate([
            'coord_lat' => 'string|required',
            'coord_long' => 'string|required',
            'coord_accu' => 'string|nullable',
            'location' => 'string|nullable',
            'coord_data' => 'nullable',
            'geocode_data' => 'nullable',
        ]);
        $user = User::getApiUser();
        // $alreadyDone = $user->attendances()->whereBetween('date', [now()->startOfDay(), now()->endOfDay()])->count() > 0;
        // if (!$alreadyDone) {
        $vals['type'] = "present";
        $vals['date'] = Carbon::now()->format(config('project.date_format'));
        $vals['signed_in_at'] = Carbon::now()->format(config('project.datetime_format'));
        // $vals['date'] = '2022-10-13';
        $attendance = $user->attendances()->create($vals);

        // "employee_attendance_log_sign_in_photo"

        if ($request->hasFile('snap') && $attendance) {
            $media = $attendance->addMedia($request->snap)
                ->toMediaCollection('employee_attendance_log_sign_in_photo');
            // $media->wasRecentlyCreated = true;
        }
        $attendance->task()->attach($request->task_id);

        return [
            'message' => 'Attendance Taken Successfully',
        ];
        // }

        // throw ValidationException::withMessages([
        //     'message' => 'Your attendance already taken for today '
        // ]);
    }


    public function closeTask(Request $request, Task $task)
    {
        $vals =   $request->validate([
            'description' => 'string|required',
            'locationInfo' => 'array|nullable',
            'snap' => 'required|image',
        ]);
        $user = User::getApiUser();
        $callReportImage = null;
        // if has file save to task/{task->id}/call-reports/
        if ($request->hasFile('snap')) {
            // $request->file('adsa')->storePublicly();
            $callReportImage = $request->snap->storePublicly("task/{$task->id}/call-reports", 'public');
        }
        $oldStatus = $task->status;

        $a = TaskEventLog::closeTaskWithCallReport(
            $task,
            $user,
            $callReportImage,
            $request->description,
            $request->locationInfo,
            $oldStatus
        );


        return $this->taskDetailed($request, $task);


        // return [
        //     'hasFile' => $request->hasFile('snap'),
        //     'all' => $request->all(),
        // ];
        // return [
        //     'message' => 'Task Closed!!',
        // ];
        // }

        // throw ValidationException::withMessages([
        //     'message' => 'Your attendance already taken for today '
        // ]);
    }


    public function MyVisits(Request $request)
    {
        $user = User::getApiUser();
        $visits =  EmployeeAttendanceLog::with([
            'user', 'task', 'task.client', 'task.clientBranch', 'task.clientGroup', 'reimbursement',
        ])
            ->withExists([
                'reimbursement',
                'task'
            ])
            ->where('user_id', $user->id)
            ->latest()
            ->paginate(50);

        return $visits;
    }

    public function addReimbursementRouteData(Request $request, EmployeeAttendanceLog $visit)
    {
        // return $request->input();
        // return $request->input('reimbursement.routes');
        $user = User::getApiUser();
        $visit->loadExists([
            'reimbursement',
            'task'
        ]);

        if ($visit->reimbursement_exists) {
            throw ValidationException::withMessages([
                'message' => 'Reimbursement data already added for this visit'
            ]);
        }
        $request->validate([
            'task_id'                                       =>      'required',
            'visit_id'                                      =>      'required',
            // 'reimbursement.amount'                          =>      'numeric',
            'reimbursement.routes.*.from'                   =>      'required',
            'reimbursement.routes.*.to'                     =>      'required',
            'reimbursement.routes.*.journey_distance'       =>      'numeric|nullable',
            'reimbursement.routes.*.journey_fare'           =>      'numeric|nullable',
            'reimbursement.routes.*.vehicle_type'           =>      'required',
            'reimbursement.routes.*.journey_type'           =>      'required',
            'reimbursement.routes.*.note'                   =>      'nullable',
            'reimbursement.routes.*.route_cost'             =>      'numeric',
        ]);

        $reimb = $visit->reimbursement()->create([
            'user_id' => $user->id,
            'task_id' => $request->get('task_id'),
            // 'amount' => $request->get('reimbursement.amount'),
        ]);

        if ($reimb) {
            $allRoutes = $request->input('reimbursement.routes');
            foreach ($allRoutes as $key => $routex) {
                $routex['user_id'] = $user->id;
                $reimb->routes()->create($routex);
            }

            $reimb->updateAmount();
        }
        $visit->load([
            'reimbursement.routes'
        ]);
        $visit->loadExists([
            'reimbursement',
            'task'
        ]);
        return $visit;


        // $visits =  EmployeeAttendanceLog::with([
        //     'user', 'task', 'task.client', 'task.clientBranch', 'task.clientGroup'
        // ])
        //     ->where('user_id', $user->id)
        //     ->latest()
        //     ->paginate(50);

        // return $visits;

    }

    public function reimbursementCfg(Request $request)
    {
        $cfg = [];
        $cfg['per_km_price'] = 2;
        $cfg['vehicle_types_assoc'] = VisitRoute::VEHICLE_TYPE_RADIO;
        $cfg['vehicle_types'] = collect(VisitRoute::VEHICLE_TYPE_RADIO)->map(function ($d, $k) {
            return [
                'value' => $k,
                'label' => $d,
            ];
        });
        $cfg['journey_types_assoc'] = VisitRoute::JOURNEY_TYPE_RADIO;
        $cfg['journey_types'] = collect(VisitRoute::JOURNEY_TYPE_RADIO)->map(function ($d, $k) {
            return [
                'value' => $k,
                'label' => $d,
            ];
        });

        return $cfg;
    }

    public function reimbursements(Request $request)
    {
        $user = User::getApiUser();
        $reimbs =  VisitReimbursement::with([
            'task',  'task.client', 'task.clientBranch', 'task.clientGroup', 'approvedBy', 'visit', 'routes',
        ])
            // ->withExists([
            //     'reimbursement',
            //     'task'
            // ])
            ->where('user_id', $user->id)
            ->latest()
            ->paginate(50);

        return $reimbs;
    }


    public function bringProductsByCategory(Request $request, ProductCategory $category)
    {
        return $category->products()->orderBy('name')->get();
    }


    public function requestProducts(Request $request, Task $task)
    {
        $vals =  $request->validate([
            'products' => 'array',
            'products.*.id' => 'required',
            'products.*.quantity' => 'required|integer',
            'products.*.description' => 'nullable',
            'products.*.unlisted' => 'nullable',

        ]);

        $prodArr = [];
        foreach ($vals['products'] as  $prod) {
            $unlisted = !!Arr::get($prod, 'unlisted');
            $prodInfo = [
                'quantity' => Arr::get($prod, 'quantity'),
                'description' => Arr::get($prod, 'description'),
                'unlisted' => $unlisted,
            ];
            if ($unlisted) {
                $prodInfo['info'] = [
                    'name' => Arr::get($prod, 'name'),
                    'model' => Arr::get($prod, 'model'),
                ];
            } else {

                $prodInfo['info'] = [
                    'name' => Arr::get($prod, 'name'),
                    'model' => Arr::get($prod, 'model'),
                ];
            }
            // if($unlisted){
            //     $prodArr[$prod['id']] = $prodInfo;
            // }
            // else {
            //     $prodArr[$prod['id']] = $prodInfo;
            // }
            $prodArr[$prod['id']] = $prodInfo;
        }


        TaskEventLog::createProductRequestEvent(
            $task,
            auth()->user(),
            $prodArr
        );

        return $this->taskDetailed($request, $task);
    }


    public function timeline(Request $request, Task $task)
    {
        $task->load([
            'taskEventLogs',
            'taskEventLogs.loggedBy',
            'taskEventLogs.assignedTo',
            'taskEventLogs.products',
        ]);
        return $task->taskEventLogs->sortByDesc('id');
        // return $this->taskDetailed($request, $task);
    }


    public function paySlips(Request $request)
    {
        $user =  auth()->user();
        $payslipsAll = $user->paySlips()->orderBy('year', 'desc')->orderBy('month', 'asc')->get();
        $paySlipGrouped =  $payslipsAll->groupBy('fin_yr_title');
        return $paySlipGrouped;
        // return $payslipsAll;
    }
}
